<?php

namespace App\Http\Controllers\User;

use App\Models\Form;
use App\Models\User;
use App\Constants\Status;
use App\Lib\FormProcessor;
use App\Models\Withdrawal;
use App\Models\DeviceToken;
use App\Models\Transaction;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use App\Lib\GoogleAuthenticator;
use App\Models\AdminNotification;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Validator;

class UserController extends Controller
{
    public function home()
    {
        $user = auth()->user();
        $pageTitle = 'Dashboard';
        $deposit = $user->successDeposits()->sum('amount');
        $transactions = $user->transactions()->orderBy('id', 'desc')->limit(8)->get();
        $commission = $user->transactions()->whereIn('remark',['referral_commission', 'level_commission'])->sum('amount');
        $withdraw = Withdrawal::where('user_id', $user->id)->where('status', Status::PAYMENT_SUCCESS)->sum('amount');
        $transaction = $user->transactions()->count();
        return view('Template::user.dashboard', compact('pageTitle', 'deposit', 'transactions', 'commission', 'withdraw', 'transaction','user'));
    }

    public function depositHistory(Request $request)
    {
        $pageTitle = 'Deposit History';
        $deposits = auth()->user()->deposits()->searchable(['trx'])->with(['gateway'])->orderBy('id','desc')->paginate(getPaginate());
        return view('Template::user.deposit_history', compact('pageTitle', 'deposits'));
    }

    public function show2faForm()
    {
        $ga = new GoogleAuthenticator();
        $user = auth()->user();
        $secret = $ga->createSecret();
        $qrCodeUrl = $ga->getQRCodeGoogleUrl($user->username . '@' . gs('site_name'), $secret);
        $pageTitle = '2FA Security';
        return view('Template::user.twofactor', compact('pageTitle', 'secret', 'qrCodeUrl'));
    }

    public function create2fa(Request $request)
    {
        $user = auth()->user();
        $request->validate([
            'key' => 'required',
            'code' => 'required',
        ]);
        $response = verifyG2fa($user,$request->code,$request->key);
        if ($response) {
            $user->tsc = $request->key;
            $user->ts = Status::ENABLE;
            $user->save();
            $notify[] = ['success', 'Two factor authenticator activated successfully'];
            return back()->withNotify($notify);
        } else {
            $notify[] = ['error', 'Wrong verification code'];
            return back()->withNotify($notify);
        }
    }

    public function disable2fa(Request $request)
    {
        $request->validate([
            'code' => 'required',
        ]);

        $user = auth()->user();
        $response = verifyG2fa($user,$request->code);
        if ($response) {
            $user->tsc = null;
            $user->ts = Status::DISABLE;
            $user->save();
            $notify[] = ['success', 'Two factor authenticator deactivated successfully'];
        } else {
            $notify[] = ['error', 'Wrong verification code'];
        }
        return back()->withNotify($notify);
    }

    public function transactions()
    {
        $pageTitle = 'Transactions';
        $remarks = Transaction::distinct('remark')->orderBy('remark')->get('remark');

        $transactions = Transaction::where('user_id',auth()->id())->searchable(['trx'])->filter(['trx_type','remark'])->orderBy('id','desc')->paginate(getPaginate());

        return view('Template::user.transactions', compact('pageTitle','transactions','remarks'));
    }

    public function kycForm()
    {
        if (auth()->user()->kv == Status::KYC_PENDING) {
            $notify[] = ['error','Your KYC is under review'];
            return to_route('user.home')->withNotify($notify);
        }
        if (auth()->user()->kv == Status::KYC_VERIFIED) {
            $notify[] = ['error','You are already KYC verified'];
            return to_route('user.home')->withNotify($notify);
        }
        $pageTitle = 'KYC Documents';
        $form = Form::where('act','kyc')->first();
        return view('Template::user.kyc.form', compact('pageTitle','form'));
    }

    public function kycData()
    {
        $user = auth()->user();
        $pageTitle = 'KYC Data';
        return view('Template::user.kyc.info', compact('pageTitle','user'));
    }

    public function kycSubmit(Request $request)
    {
        $form = Form::where('act','kyc')->firstOrFail();
        $formData = $form->form_data;
        $formProcessor = new FormProcessor();
        $validationRule = $formProcessor->valueValidation($formData);
        $request->validate($validationRule);
        $user = auth()->user();
        foreach (@$user->kyc_data ?? [] as $kycData) {
            if ($kycData->type == 'file') {
                fileManager()->removeFile(getFilePath('verify').'/'.$kycData->value);
            }
        }
        $userData = $formProcessor->processFormData($request, $formData);
        $user->kyc_data = $userData;
        $user->kyc_rejection_reason = null;
        $user->kv = Status::KYC_PENDING;
        $user->save();

        $notify[] = ['success','KYC data submitted successfully'];
        return to_route('user.home')->withNotify($notify);

    }

    public function userData()
    {
        $user = auth()->user();

        if ($user->profile_complete == Status::YES) {
            return to_route('user.home');
        }

        $pageTitle  = 'User Data';
        $info       = json_decode(json_encode(getIpInfo()), true);
        $mobileCode = @implode(',', $info['code']);
        $countries  = json_decode(file_get_contents(resource_path('views/partials/country.json')));

        return view('Template::user.user_data', compact('pageTitle', 'user', 'countries', 'mobileCode'));
    }

    public function userDataSubmit(Request $request)
    {

        $user = auth()->user();

        if ($user->profile_complete == Status::YES) {
            return to_route('user.home');
        }

        $countryData  = (array)json_decode(file_get_contents(resource_path('views/partials/country.json')));
        $countryCodes = implode(',', array_keys($countryData));
        $mobileCodes  = implode(',', array_column($countryData, 'dial_code'));
        $countries    = implode(',', array_column($countryData, 'country'));

        $request->validate([
            'country_code' => 'required|in:' . $countryCodes,
            'country'      => 'required|in:' . $countries,
            'mobile_code'  => 'required|in:' . $mobileCodes,
            'username'     => 'required|unique:users|min:6',
            'mobile'       => ['required','regex:/^([0-9]*)$/',Rule::unique('users')->where('dial_code',$request->mobile_code)],
        ]);

        if (preg_match("/[^a-z0-9_]/", trim($request->username))) {
            $notify[] = ['info', 'Username can contain only small letters, numbers and underscore.'];
            $notify[] = ['error', 'No special character, space or capital letters in username.'];
            return back()->withNotify($notify)->withInput($request->all());
        }

        $user->country_code = $request->country_code;
        $user->mobile       = $request->mobile;
        $user->username     = $request->username;


        $user->address = $request->address;
        $user->city = $request->city;
        $user->state = $request->state;
        $user->zip = $request->zip;
        $user->country_name = @$request->country;
        $user->dial_code = $request->mobile_code;

        $user->profile_complete = Status::YES;
        $user->save();

        return to_route('user.home');
    }


    public function addDeviceToken(Request $request)
    {

        $validator = Validator::make($request->all(), [
            'token' => 'required',
        ]);

        if ($validator->fails()) {
            return ['success' => false, 'errors' => $validator->errors()->all()];
        }

        $deviceToken = DeviceToken::where('token', $request->token)->first();

        if ($deviceToken) {
            return ['success' => true, 'message' => 'Already exists'];
        }

        $deviceToken          = new DeviceToken();
        $deviceToken->user_id = auth()->user()->id;
        $deviceToken->token   = $request->token;
        $deviceToken->is_app  = Status::NO;
        $deviceToken->save();

        return ['success' => true, 'message' => 'Token saved successfully'];
    }

    public function downloadAttachment($fileHash)
    {
        $filePath = decrypt($fileHash);
        $extension = pathinfo($filePath, PATHINFO_EXTENSION);
        $title = slug(gs('site_name')).'- attachments.'.$extension;
        try {
            $mimetype = mime_content_type($filePath);
        } catch (\Exception $e) {
            $notify[] = ['error','File does not exists'];
            return back()->withNotify($notify);
        }
        header('Content-Disposition: attachment; filename="' . $title);
        header("Content-Type: " . $mimetype);
        return readfile($filePath);
    }


    public function balanceTransfer()
    {
        $general = gs();
        if ($general->balance_transfer != 1) {
            $notify[] = ['warning', "Balance transfer doesn't available this time"];
            return redirect()->route('user.home')->withNotify($notify);
        }
        $pageTitle = "Balance Transfer";
        return view('Template::user.balance_transfer', compact('pageTitle'));
    }

    public function balanceTransferUser(Request $request)
    {
        $general = gs();
        if ($general->balance_transfer != 1) {
            $notify[] = ['warning', "Balance transfer doesn't available this time"];
            return redirect()->route('home')->withNotify($notify);;
        }
        $request->validate([
            'amount' => 'required|numeric|gt:0',
            'username' => 'required'
        ]);
        $user = auth()->user();
        $toUser = User::where('status', Status::VERIFIED)->where('username', $request->username)->first();
        if (!$toUser) {
            $notify[] = ['error', 'Receiver not found'];
            return back()->withNotify($notify);
        }
        if ($user->id == $toUser->id) {
            $notify[] = ['error', "You can not transfer balance to self account."];
            return back()->withNotify($notify);
        }
        $charge = (($request->amount / 100) * $general->balance_transfer_percent_charge) + $general->balance_transfer_fixed_charge;
        $total = $request->amount + $charge;

        if ($total > $user->balance) {
            $notify[] = ['error', 'Your account balance ' . getAmount($user->balance) . ' ' . $general->cur_text . ' not enough for balance transfer'];
            return back()->withNotify($notify);
        }
        $user->balance -=  $total;
        $user->save();

        $transaction = new Transaction();
        $transaction->user_id = $user->id;
        $transaction->amount = $total;
        $transaction->post_balance = $user->balance;
        $transaction->charge = $charge;
        $transaction->trx_type = '-';
        $transaction->details = 'Balance Transferred To ' . $toUser->username;
        $transaction->remark = "balance_send";
        $transaction->trx = getTrx();
        $transaction->save();

        notify($user, 'BAL_TRANSFER_SENDER', [
            'trx' => $transaction->trx,
            'amount' => showAmount($request->amount,currencyFormat:false),
            'charge' => showAmount($charge,currencyFormat:false),
            'after_charge' => showAmount($total,currencyFormat:false),
            'post_balance' => showAmount($user->balance,currencyFormat:false),
        ]);

        $toUser->balance += $request->amount;
        $toUser->save();

        $tranUser = new Transaction();
        $tranUser->user_id = $toUser->id;
        $tranUser->amount = $request->amount;
        $tranUser->post_balance = $toUser->balance;
        $tranUser->trx_type = '+';
        $tranUser->details = 'Balance Transferred From ' . $user->username;
        $tranUser->trx = $transaction->trx;
        $tranUser->remark = "balance_receive";
        $tranUser->save();

        notify($toUser, 'BAL_TRANSFER_RECEIVER', [
            'trx' => $tranUser->trx,
            'amount' => showAmount($tranUser->amount,currencyFormat:false),
            'post_balance' => showAmount($toUser->balance,currencyFormat:false),
        ]);

        $adminNotification = new AdminNotification();
        $adminNotification->user_id = $user->id;
        $adminNotification->title = 'Balance Transferred To ' . $toUser->username;
        $adminNotification->click_url = urlPath('admin.report.transaction');
        $adminNotification->save();

        $adminNotification = new AdminNotification();
        $adminNotification->user_id = $user->id;
        $adminNotification->title = 'Balance Transferred From ' . $user->username;
        $adminNotification->click_url = urlPath('admin.report.transaction');
        $adminNotification->save();

        $notify[] = ['success', 'Balance has been transfer'];
        return back()->withNotify($notify);
    }

    public function referralLog()
    {
        $pageTitle = 'My Referred Users';
        $user      = auth()->user();
        return view('Template::user.referral', compact('pageTitle', 'user'));
    }

    public function referralCommission()
    {
        $user = auth()->user();
        $pageTitle = "My Referral Commissions";
        $commissions = $user->transactions()->where('remark','referral_commission')->paginate(getPaginate());
        return view('Template::user.commission', compact('pageTitle', 'commissions'));
    }

    public function levelCommission()
    {
        $user = auth()->user();
        $pageTitle = "My Level Commissions";
        $commissions = $user->transactions()->where('remark','level_commission')->paginate(getPaginate());
        return view('Template::user.commission', compact('pageTitle', 'commissions'));
    }

}
